/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.utils.file_transfer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Callable;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import gov.vha.isaac.utils.file_transfer.FileTransfer.FileActionResult;

/**
 * 
 * {@link Upload}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class Upload implements Callable<URL>
{
	private static Logger log = LoggerFactory.getLogger(Upload.class);
	String username_, password_;
	String groupId_, artifactId_, version_, classifier_, type_;
	String targetRepositoryRootUrl_;
	String relativeFilePath_ = null;
	URL destinationUrl_ = null;
	boolean overwrite_;
	boolean stripSnapshotVersionDetails_;
	String literalFileBasename_ = null;

	/**
	 * @param username (optional) used if provided
	 * @param password (optional) used if provided
	 * @param targetRepositoryRootUrl The URL to upload to
	 * @param groupId
	 * @param artifactId
	 * @param version
	 * @param classifier
	 * @param type
	 * @throws IOException
	 */
	public Upload(
			String username,
			String password,
			String targetRepositoryRootUrl,
			String groupId, String artifactId, String version, String classifier, String type,
			boolean stripSnapshotVersionDetails,
			boolean overwrite) throws IOException
	{
		this.username_ = username;
		this.password_ = password;
		this.targetRepositoryRootUrl_ = targetRepositoryRootUrl;

		this.groupId_ = groupId;
		this.artifactId_ = artifactId;
		this.version_ = version;
		this.classifier_ = classifier;
		this.type_ = type;
		
		this.stripSnapshotVersionDetails_ = stripSnapshotVersionDetails;
		this.overwrite_ = overwrite;
	}

	public Upload(
			String username,
			String password,
			String targetRepositoryRootUrl,
			String groupId, String artifactId, String version,
			String literalFileBasename,
			boolean overwrite) throws IOException
	{
		this.username_ = username;
		this.password_ = password;
		this.targetRepositoryRootUrl_ = targetRepositoryRootUrl;

		this.groupId_ = groupId;
		this.artifactId_ = artifactId;
		this.version_ = version;
		this.classifier_ = null;
		this.type_ = null;
		
		this.stripSnapshotVersionDetails_ = false;
		this.overwrite_ = overwrite;
		this.literalFileBasename_ = literalFileBasename;
	}

	public Set<FileTransferFutureTask<URL>> generateSubTasks() throws MalformedURLException, IOException, Exception {
		Set<FileTransferFutureTask<URL>> subtasks = new HashSet<>();
		
		File sha1File = new File(getRelativeFilePath() + ".sha1");
		// Only try to upload SHA1 file if it exists
		if (sha1File.exists()) {
			URL destinationUrl = new URL(getDestinationUrl() + ".sha1");
			subtasks.add(new FileTransferFutureTask<URL>(new SimpleUpload(username_, password_, sha1File, destinationUrl, overwrite_), "simple upload of " + sha1File.getAbsolutePath() + " to " + destinationUrl));
		}

		File md5File = new File(getRelativeFilePath() + ".md5");
		// Only try to upload MD5 file if it exists
		if (md5File.exists()) {
			URL destinationUrl = new URL(getDestinationUrl() + ".md5");
			subtasks.add(new FileTransferFutureTask<URL>(new SimpleUpload(username_, password_, md5File, destinationUrl, overwrite_), "simple upload of " + md5File.getAbsolutePath() + " to " + destinationUrl));
		}
		
		return Collections.unmodifiableSet(subtasks);
	}
	public URL getDestinationUrl() throws MalformedURLException, Exception {
		if (destinationUrl_ == null) {
			destinationUrl_ = new URL(targetRepositoryRootUrl_ + (targetRepositoryRootUrl_.endsWith("/") ? "" : "/") + getRelativeFilePath());
		}
		
		return destinationUrl_;
	}
	String getRelativeFilePath() throws Exception {
		if (relativeFilePath_ == null) {
			if (literalFileBasename_ == null) {
				relativeFilePath_ = getRelativeFilePath(groupId_, artifactId_, version_, classifier_, type_, stripSnapshotVersionDetails_);
			} else {
				relativeFilePath_ = getRelativeFilePath(groupId_, artifactId_, version_, literalFileBasename_);
			}
		}
		
		return relativeFilePath_;
	}
	private static String getRelativeFilePath(
			String groupId, String artifactId, String version,
			String fileBasename) throws Exception {
		String pathFromGroupId = groupId.replaceAll("\\.", "/");
		
		String relativePath = pathFromGroupId + "/" + artifactId + "/" + version;
		
		return relativePath + "/" + fileBasename;
	}
	private static String getRelativeFilePath(
			String groupId, String artifactId, String version, String classifier,
			String type,
			boolean stripSnapshotVersionDetails) throws Exception {
		String pathFromGroupId = groupId.replaceAll("\\.", "/");
		String snapshotVersion = "";
		String versionWithoutSnapshot = version;

		// Handling of timestamped snapshots
		String relativePath = pathFromGroupId + "/" + artifactId + "/" + version;
		if (version.endsWith("-SNAPSHOT") && ! stripSnapshotVersionDetails) {
			versionWithoutSnapshot = version.substring(0, version.lastIndexOf("-SNAPSHOT"));

			File metadataFile = new File(relativePath + "/" + "maven-metadata.xml");
			
			DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder;
			Document dDoc = null;
			XPath xPath = XPathFactory.newInstance().newXPath();

			builder = domFactory.newDocumentBuilder();

			dDoc = builder.parse(metadataFile);
			String timestamp = ((Node)xPath.evaluate("/metadata/versioning/snapshot/timestamp", dDoc, XPathConstants.NODE)).getTextContent();
			String buildNumber = ((Node)xPath.evaluate("/metadata/versioning/snapshot/buildNumber", dDoc, XPathConstants.NODE)).getTextContent();
			
			snapshotVersion = "-" + timestamp + "-" + buildNumber;
		}
		
		return pathFromGroupId + "/" + artifactId + "/" + version + "/" + artifactId + "-" + versionWithoutSnapshot + snapshotVersion +
				(StringUtils.isNotBlank(classifier) ? "-" + classifier : "") + "." + type;
	}
	
	public static URL upload(File file, URL destinationUrl, String username, String password, boolean overwrite) throws Exception
	{
		OutputStream out = null;
		FileInputStream fis = null;
		HttpURLConnection httpCon = null;
		try
		{
			if (FileTransfer.hasBeenSuccessfullyHandled(file.getAbsolutePath()) || (! overwrite && FileTransferUtils.remoteFileExists(destinationUrl, username, password))) {
				log.debug("Not uploading already uploaded file " + file.getAbsolutePath() + " to " + destinationUrl.toString());
				
				if (! FileTransfer.hasBeenSuccessfullyHandled(file.getAbsolutePath())) {
					FileTransfer.setFileActionResult(file.getAbsolutePath(), FileActionResult.EXISTS);
				}

				return destinationUrl;
			} else {
				FileTransfer.setFileActionResult(file.getAbsolutePath(), FileActionResult.FAILED);
			}
			log.debug("Uploading " + file.getAbsolutePath() + " to " + destinationUrl.toString());

			httpCon = NetworkUtils.getConnection(destinationUrl, username, password);
			httpCon.setDoOutput(true);
			httpCon.setRequestMethod("PUT");
			httpCon.setConnectTimeout(30 * 1000);
			httpCon.setReadTimeout(60 * 60 * 1000);
			//httpCon.setChunkedStreamingMode(8192);
			long fileLength = file.length();
			httpCon.setFixedLengthStreamingMode(fileLength);
			out = httpCon.getOutputStream();

			int bytesReadSinceShowingProgress = 0;
			byte[] buf = new byte[8192];
			fis = new FileInputStream(file);
			int read = 0;
			while ((read = fis.read(buf, 0, buf.length)) > 0)
			{
				if (Thread.interrupted()) {
					throw new InterruptedException();
				}
				out.write(buf, 0, read);

				bytesReadSinceShowingProgress += buf.length;
				if (bytesReadSinceShowingProgress >= FileTransfer.IO_SHOW_PROGRESS_BYTES_THRESHOLD) {
					ConsoleUtil.showProgress(); // Show progress for buffered I/O
					bytesReadSinceShowingProgress = 0;
				}
				
				if (FileTransfer.DEBUG_FAIL_PROCESSING_FILE_NAME != null && file.getName().equals(FileTransfer.DEBUG_FAIL_PROCESSING_FILE_NAME)) {
					throw new IOException("intentionally failing upload of \"" + file.getName() + "\" for debugging purposes");
				}
			}

			out.flush();
			
			log.debug("Done uploading " + file.getAbsolutePath() + " to " + destinationUrl.toString());

			FileTransfer.setFileActionResult(file.getAbsolutePath(), FileActionResult.SUCCEEDED);

			StringBuilder sb = new StringBuilder();
			try {
				InputStream is = httpCon.getInputStream();
				read = 0;
				byte[] buffer = new byte[1024];
				CharBuffer cBuffer = ByteBuffer.wrap(buffer).asCharBuffer();
				while (read != -1)
				{
					read = is.read(buffer);
					if (read > 0)
					{
						sb.append(cBuffer, 0, read);
					}
				}
			} catch (IOException e) {
				log.debug("Unable to read server response message after uploading file " + file.getAbsolutePath() + ") to destination (" + destinationUrl + ")");
			}
			if (sb.toString().trim().length() > 0)
			{
				throw new Exception("The server reported an error during the publish operation of file " + file.getAbsolutePath() + ":  " + sb.toString());
			}
		}
		catch (Exception e1)
		{
			log.error("Failed to upload file (" + file.getAbsolutePath() + ") to destination (" + destinationUrl + ")", e1);
			FileTransfer.setFileActionResult(file.getAbsolutePath(), FileActionResult.FAILED);
			throw e1;
		}
		finally {
			if (httpCon != null) {
				httpCon.disconnect();
			}
			if (out != null) {
				out.close();
			}
			if (fis != null) {
				fis.close();
			}
		}

		return destinationUrl;
	}

	/* (non-Javadoc)
	 * @see java.util.concurrent.Callable#call()
	 */
	public URL call() throws Exception
	{
		URL destinationUrl = null;
		String relativeFile = null;

		relativeFile = getRelativeFilePath();
		destinationUrl = getDestinationUrl();

		File file = new File(relativeFile);

		return upload(file, destinationUrl, username_, password_, overwrite_);
	}
}